function [E,H] = far_fields(pos, alphae, alphah, alphaeh, EHsol, k0, ...
    posEv, epsout)

% Optimized function for obtaining scattered electric and magnetic far 
% field.

%%%%%% INPUT ARGUMENTS %%%%%%
%%% pos:     position of the dipoles                                 %%%
%%% alphae:  electric polarizabilities of the dipoles                %%%
%%% alphah:  magnetic polarizabilities of the dipoles                %%%
%%% alphaeh: mixed magnetic electric polarizabilities of the dipoles %%%
%%% EHsol:   total exciting electromagnetic field at each dipole     %%%
%%% k0:      wavenumber                                              %%%
%%% posEv:   position where the EM field wants to be calculated      %%%
%%% espout:  surrounding permittivity                                %%%


%%%%%% OUTPUT ARGUMENTS %%%%%%
%%% E: electric scattered far field %%%
%%% H: magnetic scattered far field %%%

% Constants %
load('vacPer.mat','eps0','mu0');
eps0 = epsout*eps0;
Z0   = sqrt(mu0/eps0);

% Auxiliary data % 
Rd = norm(posEv); 
nd = posEv/Rd;
M  = (1/eps0)*(eye(3)-nd'*nd);

% Data for loops %
N  = length(pos(:,1));
K  = length(k0);
E  = zeros(K,3);
H  = E;

% If L = 2, the polarizability is wavelength indepndent and the same for
% all dipole. If K>1 and L = 3 the polarizability is only a function of
% the wavelength and if L = 4 it also depends on the position (dipole).
% If K = 1 and L = 3 it only depndends on the position and if L = 2 the
% polarizability is wavelength independent and the same for all dipoles.
S = size(alphae);
L = length(S);

if(K > 1)
    if(L > 3)
        parfor i = 1:K

            EHinc = squeeze(EHsol(i,:,:));
            En = 0;
            Hn = 0;

            for j = 1:N

                ae  = squeeze(alphae(i,j,:,:));
                ah  = squeeze(alphah(i,j,:,:));
                aeh = squeeze(alphaeh(i,j,:,:));
                En = En + exp(-1i*k0(i)*dot(nd,pos(i,:)))*...
                    (ae*(EHinc(i,1:3)).' - 1i*mu0*aeh*(EHinc(i,4:6)).');
                Hn = Hn + exp(-1i*k0(i)*dot(nd,pos(i,:)))*...
                    (ah*(EHinc(i,4:6)).' + 1i*aeh*(EHinc(i,1:3)).');

            end

            E(i,:)   = (k0(i)^2*exp(1i*k0(i)*Rd)/(4*pi*Rd))*(M*En - ...
                Z0*cross(nd.',Hn)).';
            H(i,:)   = (1/Z0)*cross(nd,E(i,:));

        end
    else
        parfor i = 1:K
        
            En = 0;
            Hn = 0;
            EHinc = squeeze(EHsol(i,:,:));
            ae    = squeeze(alphae(i,:,:));
            ah    = squeeze(alphah(i,:,:));
            aeh   = squeeze(alphaeh(i,:,:));

            for j = 1:N
                En = En + exp(-1i*k0(i)*dot(nd,pos(j,:)))*...
                    (ae*(EHinc(j,1:3)).' - 1i*mu0*aeh*(EHinc(j,4:6)).');
                Hn = Hn + exp(-1i*k0(i)*dot(nd,pos(j,:)))*...
                    (ah*(EHinc(j,4:6)).' + 1i*aeh*(EHinc(j,1:3)).');
            end

            E(i,:)   = (k0(i)^2*exp(1i*k0(i)*Rd)/(4*pi*Rd))*(M*En - ...
                Z0*cross(nd.',Hn)).';
            H(i,:)   = (1/Z0)*cross(nd,E(i,:));
            
        end
    end
   
else
    
    En = 0;
    Hn = 0;
    EHinc = squeeze(EHsol);

    if(L > 2)

        for i = 1:N

            ae  = squeeze(alphae(i,:,:));
            ah  = squeeze(alphah(i,:,:));
            aeh = squeeze(alphaeh(i,:,:));
            En = En + exp(-1i*k0*dot(nd,pos(i,:)))*...
                (ae*(EHinc(i,1:3)).' - 1i*mu0*aeh*(EHinc(i,4:6)).');
            Hn = Hn + exp(-1i*k0*dot(nd,pos(i,:)))*...
                (ah*(EHinc(i,4:6)).' + 1i*aeh*(EHinc(i,1:3)).');

        end

        E(:)   = (k0^2*exp(1i*k0*Rd)/(4*pi*Rd))*(M*En - Z0*cross(nd.',Hn)).';
        H(:)   = (1/Z0)*cross(nd,E(:));

    else
    
        for i = 1:N
            En = En + exp(-1i*k0*dot(nd,pos(i,:)))*(alphae*(EHinc(i,1:3)).' ...
                - 1i*mu0*alphaeh*(EHinc(i,4:6)).');
            Hn = Hn + exp(-1i*k0*dot(nd,pos(i,:)))*(alphah*(EHinc(i,4:6)).' ...
                + 1i*alphaeh*(EHinc(i,1:3)).');
        end

        E(:)   = (k0^2*exp(1i*k0*Rd)/(4*pi*Rd))*(M*En - Z0*cross(nd.',Hn)).';
        H(:)   = (1/Z0)*cross(nd,E(:));

    end
    
end


end
